// src/user-panel/OrderBuilder.ts

const COMPANY_ID = process.env.NEXT_PUBLIC_SUMIT_COMPANY_ID;
const API_KEY = process.env.NEXT_PUBLIC_SUMIT_TOKEN;

export interface User {
  CustomerName: string;
  num_of_members: number;
  [key: string]: any;
}

export interface OrderItem {
  ItemID: number;
  Name?: string;
  Quantity: number;
  Editable: boolean;
}

export const getCustomerDocuments = async (CustomerName: string): Promise<any[]> => {
  let allDocs: any[] = [];
  let page = 0;
  const pageSize = 100;
  while (true) {
    const res = await fetch('https://api.sumit.co.il/accounting/documents/list/', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        Credentials: { CompanyID: COMPANY_ID, APIKey: API_KEY },
        IncludeDrafts: false,
        DateFrom: '2023-01-01T00:00:00Z',
        DateTo: new Date().toISOString(),
        Paging: { StartIndex: page * pageSize, PageSize: pageSize },
      }),
    });
    const json = await res.json();
    const docs = json?.Data?.Documents || [];
    allDocs = allDocs.concat(docs);
    if (docs.length < pageSize) break;
    page++;
  }
  return allDocs.filter((doc: any) => doc.CustomerName === CustomerName) || [];
};

export const getAllDocumentItems = async (documents: any[]): Promise<any[]> => {
  const results: any[] = [];
  for (const doc of documents) {
    const res = await fetch('https://api.sumit.co.il/accounting/documents/getdetails/', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        Credentials: { CompanyID: COMPANY_ID, APIKey: API_KEY },
        DocumentID: doc.DocumentID,
        DocumentType: doc.Type || 0,
      }),
    });
    const json = await res.json();
    if (json?.Data?.Items) {
      results.push(
        ...json.Data.Items.map((item: any) => ({
          ItemID: item.Item.ID,
          Name: item.Item.Name,
          Date: doc.Date,
          Quantity: item.Quantity,
        }))
      );
    }
  }
  return results;
};

const checkHistory = (items: any[], targetID: number, currentDate: Date): { included: boolean; monthsSince: number } => {
  const dates = items
    .filter((item) => item.ItemID === targetID)
    .map((i) => new Date(i.Date))
    .sort((a, b) => b.getTime() - a.getTime());

  if (!dates.length) return { included: true, monthsSince: 0 };

  const lastDate = dates[0];
  const sameMonth = lastDate.getMonth() === currentDate.getMonth() && lastDate.getFullYear() === currentDate.getFullYear();

  if (sameMonth) return { included: false, monthsSince: 0 };

  const monthsSince =
    (currentDate.getFullYear() - lastDate.getFullYear()) * 12 +
    currentDate.getMonth() - lastDate.getMonth() +
    1;

  return { included: true, monthsSince };
};

export const fetchIncomeItems = async (): Promise<any[]> => {
  const res = await fetch('https://api.sumit.co.il/accounting/incomeitems/list/', {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      Credentials: { CompanyID: COMPANY_ID, APIKey: API_KEY },
      Paging: { StartIndex: 0, PageSize: 100 },
    }),
  });
  const json = await res.json();
  return json?.Data?.IncomeItems || [];
};

export const buildOrder = (items: any[], user: User): OrderItem[] => {
  const now = new Date();
  const result: OrderItem[] = [];

  // 1. 431127221 — всегда в заказе
  result.push({ ItemID: 431127221, Quantity: 1, Editable: true });

  // Helper: был ли товар оплачен в этом месяце?
  const wasPaidThisMonth = (targetID: number) => {
    return items.some(item => {
      if (item.ItemID !== targetID) return false;
      const d = new Date(item.Date);
      return d.getMonth() === now.getMonth() && d.getFullYear() === now.getFullYear();
    });
  };

  // Helper: получить все оплаты по ID и по "массивному" товару
  const getAllPaidQuantities = (targetID: number) => {
    // Обычные оплаты по ID
    let paid = items.filter(item => item.ItemID === targetID).map(item => ({ date: item.Date, quantity: item.Quantity || 1 }));
    // Оплаты по "массивному" товару
    const massItems = items.filter(item => item.Name === "מיסים והיטל ביטחון");
    for (const mass of massItems) {
      // Цена одной единицы для каждого из двух товаров
      // (цены должны быть известны заранее, например, из incomeItems)
      // Здесь предполагаем, что Quantity относится к обоим товарам
      paid.push({ date: mass.Date, quantity: mass.Quantity || 1 });
    }
    return paid;
  };

  // Helper: месяцев с последней оплаты (или с первого месяца), с учётом "массивного" товара
  const monthsToPay = (targetID: number) => {
    const paidArr = getAllPaidQuantities(targetID)
      .sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());
    if (!paidArr.length) return 1;
    // Дата первого платежа
    const firstDate = new Date(paidArr[0].date);
    const monthsTotal =
      (now.getFullYear() - firstDate.getFullYear()) * 12 +
      now.getMonth() - firstDate.getMonth() + 1;
    // Сумма всех quantity (учитываем авансовые оплаты)
    const paidMonths = paidArr.reduce((sum, item) => sum + (item.quantity || 1), 0);
    const diff = monthsTotal - paidMonths;
    return diff > 0 ? diff : 0;
  };

  // 2. 431126765
  if (!wasPaidThisMonth(431126765)) {
    result.push({ ItemID: 431126765, Quantity: user.num_of_members, Editable: false });
  }

  // 3. 438105061
  if (!wasPaidThisMonth(438105061)) {
    const months = monthsToPay(438105061);
    if (months > 0) {
      result.push({ ItemID: 438105061, Quantity: months, Editable: false });
    }
  }

  // 4. 519135662
  if (!wasPaidThisMonth(519135662)) {
    const months = monthsToPay(519135662);
    if (months > 0) {
      result.push({ ItemID: 519135662, Quantity: months, Editable: false });
    }
  }

  return result;
}; 