"use client";

import React, { useState, useEffect, useCallback } from "react";
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { Skeleton } from '../components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '../components/ui/alert';
import { AlertCircle, BellOff } from "lucide-react";
import { useTranslation } from 'react-i18next';
import {
  ComposedChart,
  Area,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  Legend,
  ResponsiveContainer,
  AreaChart
} from "recharts";
import { useAuth } from '../hooks/use-auth';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '../components/ui/select';
import GaugeChart from './GaugeChart';
import { Button } from '../components/ui/button';

interface ApiDataPoint {
  _time: string;
  _value: number | null; // This is the primary meter reading after API processing
  batteryVoltage?: number | null;
  valveStatus?: string | null;
  deviceName?: string;
  dev_eui?: string;
  device_type?: string;
}

interface DeviceInfo {
  device_id: string;
  dev_eui: string;
  device_type: string;
}

interface ValveStatus {
  icon: string;
  label: string;
}

interface DashboardTabProps {
  isSystem: boolean;
}

// Кастомный тултип для AreaChart
const CustomUsageTooltip = ({ active, payload, label, deviceType }: any) => {
  if (active && payload && payload.length) {
    const value = payload[0].value;
    const unit = deviceType === 'Electric meter' ? 'kW' : 'm³';
    return (
      <div className="rounded bg-background border px-3 py-2 shadow text-xs">
        <div className="mb-1 font-medium">{label}</div>
        <div>
          <span className="font-bold">{Number(value).toFixed(2)} {unit}</span>
        </div>
      </div>
    );
  }
  return null;
};

// Кастомный компонент для вертикальных меток под осью X
const VerticalTick = (props: any) => {
  const { x, y, payload } = props;
  return (
    <g transform={`translate(${x},${y + 50})`}>
      <text x={0} y={0} dy={0} dx={-8} textAnchor="end" fontSize={12} transform="rotate(-90)">
        {payload.value}
      </text>
    </g>
  );
};

const DashboardTab: React.FC<DashboardTabProps> = ({ isSystem }) => {
  const { t, i18n } = useTranslation();
  
  const durationOptions = [
    { value: '1h', label: t('dashboard.lastHour', 'Last hour') },
    { value: '6h', label: t('dashboard.last6Hours', 'Last 6 hours') },
    { value: '24h', label: t('dashboard.last24Hours', 'Last 24 hours') },
    { value: '7d', label: t('dashboard.last7Days', 'Last 7 days') },
    { value: '30d', label: t('dashboard.last30Days', 'Last 30 days') },
    { value: '365d', label: t('dashboard.last365Days', 'Last 365 days') },
  ];
  const [devices, setDevices] = useState<DeviceInfo[]>([]);
  const [barData, setBarData] = useState<Record<string, { month: string; usage: number }[]>>({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [valveStatuses, setValveStatuses] = useState<Record<string, ValveStatus | null>>({});
  const { user, userPermissions, userRoleId } = useAuth();
  const [windowWidth, setWindowWidth] = useState<number>(typeof window !== 'undefined' ? window.innerWidth : 1200);

  // Расширенный маппинг типа устройства к ключу перевода
  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
    // Добавьте другие типы устройств по мере необходимости
  };

  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  // Fetch all user devices
  const fetchUserDevices = useCallback(async () => {
    if (!user?.email) {
      setError('User information not available');
      setLoading(false);
      return;
    }
    try {
      const response = await fetch(`/api/postgres/user-device?email=${encodeURIComponent(user.email)}`);
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || 'Failed to fetch user devices');
      }
      const data = await response.json();
      setDevices(data.devices || []);
    } catch (error: any) {
      setError(error.message || 'Failed to fetch user devices');
      setDevices([]);
    } finally {
      setLoading(false);
    }
  }, [user]);

  // Fetch bar chart data for all devices
  const fetchAllBarData = useCallback(async (devices: DeviceInfo[]) => {
    setLoading(true);
    setError(null);
    const newBarData: Record<string, { month: string; usage: number }[]> = {};
    for (const device of devices) {
      try {
        const response = await fetch('/api/postgres/device-history', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ deviceName: device.device_id })
        });
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || 'Failed to fetch device history');
        }
        const data = await response.json();
        newBarData[device.device_id] = data.usage || [];
      } catch (error: any) {
        setError(error.message || 'Failed to fetch device history');
        newBarData[device.device_id] = [];
        }
      }
    setBarData(newBarData);
    setLoading(false);
  }, []);

  // Handle valve status change from GaugeChart
  const handleValveStatusChange = useCallback((deviceId: string, status: ValveStatus | null) => {
    setValveStatuses(prev => ({
      ...prev,
      [deviceId]: status
    }));
  }, []);

  useEffect(() => {
    fetchUserDevices();
  }, [fetchUserDevices]);

  useEffect(() => {
    if (devices.length > 0) {
      fetchAllBarData(devices);
    }
  }, [devices, fetchAllBarData]);

  useEffect(() => {
    const handleResize = () => setWindowWidth(window.innerWidth);
    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  // Helper to process bar data for chart (usage difference, label formatting)
  const getProcessedBarData = (rawData: { month: string; usage: number }[], deviceType: string) =>
    rawData
      .map((entry, idx, arr) => {
        if (idx === 0) return null;
        const prev = arr[idx - 1];
        let usage = entry.usage - prev.usage;
        if (usage < 0) usage = 0;
        const [mm, yyyy] = entry.month.split('_');
        const date = new Date(Number(yyyy), Number(mm) - 1);
        let label = '';
        if (windowWidth <= 900) {
          // sm/md: 07.2024
          label = `${mm.padStart(2, '0')}.${yyyy}`;
        } else {
          // desktop: локализованное название месяца
          label = date.toLocaleString(i18n.language, { month: 'short', year: 'numeric' });
        }
        return { ...entry, usage, label };
      })
      .filter(Boolean);

  return (
    <div className="grid gap-6 sm:gap-4 sm:px-2 md:px-0">
      {devices.length === 0 && !loading && (
        <p className="text-muted-foreground text-center py-4 text-sm sm:text-xs">{t('dashboard.noDevices', 'No devices found for this user.')}</p>
      )}
      {devices.map((device, idx) => (
        <div key={`${device.device_id || device.dev_eui || 'device'}_${idx}`} className="grid md:grid-cols-5 gap-6 sm:gap-3 mb-8 sm:mb-4">
          <Card className="md:col-span-2 sm:col-span-5">
            <CardHeader className="px-3 sm:px-4 md:px-6">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <CardTitle className="text-base sm:text-sm">
                    {getDeviceTypeLabel(device.device_type)}
                    : {device.device_id}
                  </CardTitle>
                  <p className="text-xs text-muted-foreground">{t('dashboard.deveui', 'DEVEUI')}: {device.dev_eui}</p>
                </div>
                {valveStatuses[device.device_id] && (
                  <div className="flex flex-col items-center ml-4 sm:ml-6 md:ml-8">
                    <img 
                      src={valveStatuses[device.device_id]!.icon} 
                      alt="Valve Status" 
                      className="w-6 h-6 md:w-8 md:h-8"
                    />
                    <span className="text-xs font-medium mt-1">
                      {valveStatuses[device.device_id]!.label}
                    </span>
                  </div>
                )}
              </div>
            </CardHeader>
            <CardContent>
              <GaugeChart
                deviceName={device.device_id}
                devEUI={device.dev_eui}
                unit={device.device_type === 'Electric meter' ? 'kWh' : 'm³'}
                onValveStatusChange={(status) => handleValveStatusChange(device.device_id, status)}
              />
              {error && (
                <Alert variant="destructive" className="mt-4">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>{t('errors.loadingStatus', 'Error Loading Status')}</AlertTitle>
                  <AlertDescription>{error.split('\n')[0]}</AlertDescription>
                </Alert>
              )}
            </CardContent>
          </Card>
          <Card className="md:col-span-3 sm:col-span-5">
            <CardHeader>
              <CardTitle className="text-base sm:text-sm">{t('dashboard.usageTrend', 'Usage Trend (Monthly)')}</CardTitle>
              <p className="text-xs text-muted-foreground pt-1">{t('dashboard.device', 'Device')}: {device.device_id}</p>
            </CardHeader>
            <CardContent>
              {loading ? (
                <Skeleton className="w-full" />
              ) : error ? (
                <Alert variant="destructive">
                  <AlertCircle className="h-4 w-4" />
                  <AlertTitle>{t('errors.fetchingTrendData', 'Error Fetching Trend Data')}</AlertTitle>
                  <AlertDescription>{error}</AlertDescription>
                </Alert>
              ) : (barData[device.device_id] && getProcessedBarData(barData[device.device_id], device.device_type).length > 0) ? (
                <ResponsiveContainer width="100%" height={300}>
                  <AreaChart
                    data={getProcessedBarData(barData[device.device_id], device.device_type)}
                    margin={{
                      top: 5,
                      right: 0,
                      bottom: 60,
                      left: -10,
                    }}
                  >
                    <defs>
                      <linearGradient id="usageGradient" x1="0" y1="0" x2="0" y2="1">
                        <stop offset="0%" stopColor="#2196f3" stopOpacity={0.5} />
                        <stop offset="100%" stopColor="#2196f3" stopOpacity={0.1} />
                      </linearGradient>
                    </defs>
                    <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                    <XAxis
                      dataKey="label"
                      //angle={270}
                      //textAnchor="start"
                      height={80}
                      axisLine={false}
                      tickLine={false}
                      tick={<VerticalTick />}
                    />
                    <YAxis tick={{ fontSize: 12 }} minTickGap={0} axisLine={false} tickLine={false} />
                    <Tooltip
                      content={<CustomUsageTooltip deviceType={device.device_type} />}
                      cursor={{ stroke: '#2196f3', strokeWidth: 1, opacity: 0.2 }}
                    />
                    <Area
                      type="monotone"
                      dataKey="usage"
                      stroke="#43a047"
                      strokeWidth={2}
                      fill="url(#usageGradient)"
                      dot={{ r: 3, stroke: '#43a047', strokeWidth: 2, fill: '#fff' }}
                      activeDot={{ r: 5 }}
                    />
                  </AreaChart>
                </ResponsiveContainer>
              ) : (
                <div className="flex items-center justify-center h-[300px] text-muted-foreground">
                  <div className="text-center">
                    <BellOff className="h-8 w-8 mx-auto mb-2 opacity-50" />
                    <p className="text-sm">No trend data available</p>
                  </div>
                </div>
              )}
            </CardContent>
          </Card>
        </div>
      ))}
    </div>
  );
};

export default DashboardTab;
