import React from 'react';
import * as Tabs from '@radix-ui/react-tabs';
import { useState } from 'react';
import { useTranslation } from 'react-i18next';
import { useAuth } from '../hooks/use-auth';

export interface TabItem {
  id: string;
  label: string;
}

interface ResponsiveTabsProps {
  tabs: TabItem[];
  activeTab: string;
  setActiveTab: (id: string) => void;
}

export default function ResponsiveTabs({ tabs, activeTab, setActiveTab }: ResponsiveTabsProps) {
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const { t, i18n } = useTranslation();
  const { user, orderTab } = useAuth();

  // Фильтруем табы по orderTab
  const filteredTabs = tabs.filter(tab => tab.id !== 'order' || orderTab !== false);

  const tabsToRender = i18n.language === 'he' ? filteredTabs.slice().reverse() : filteredTabs;

  return (
    <div className="w-full">
      {/* Desktop Tabs */}
      <Tabs.Root value={activeTab} onValueChange={setActiveTab} className="hidden lg:flex w-full border-b">
        <Tabs.List
          className={`flex w-full ${i18n.language === 'he' ? 'justify-end' : ''}`}
          dir={i18n.language === 'he' ? 'rtl' : 'ltr'}
          style={i18n.language === 'he' ? { flexDirection: 'row-reverse' } : {}}
        >
          {tabsToRender.map(tab => (
            <Tabs.Trigger
              key={tab.id}
              value={tab.id}
              className={`flex-1 px-2 sm:px-4 py-2 sm:py-3 text-center font-medium transition-colors text-sm sm:text-base
                ${activeTab === tab.id ? 'border-b-2 border-blue-600 text-black bg-[#A3D9F2]' : 'text-gray-500 hover:text-blue-600'}
              `}
            >
              {t(`navigation.${tab.id}`, tab.label)}
            </Tabs.Trigger>
          ))}
        </Tabs.List>
      </Tabs.Root>

      {/* Mobile Dropdown */}
      <div className="lg:hidden w-full relative">
        <button
          className="w-full px-3 py-2 sm:px-4 sm:py-3 bg-white border rounded shadow flex justify-between items-center text-sm sm:text-base"
          onClick={() => setDropdownOpen(open => !open)}
        >
          {filteredTabs.find(tab => tab.id === activeTab) ? t(`navigation.${filteredTabs.find(tab => tab.id === activeTab)!.id}`, filteredTabs.find(tab => tab.id === activeTab)!.label) : t('common.menu', 'Menu')}
          <span className="ml-2">{dropdownOpen ? "▲" : "▼"}</span>
        </button>
        {dropdownOpen && (
          <div className="absolute w-full bg-white border rounded-b shadow z-10 max-h-60 overflow-y-auto">
            {filteredTabs.map(tab => (
              <button
                key={tab.id}
                className={`w-full text-left px-3 py-2 sm:px-4 sm:py-3 hover:bg-gray-100 text-sm sm:text-base ${
                  activeTab === tab.id ? "font-bold text-blue-600" : ""
                }`}
                onClick={() => {
                  setActiveTab(tab.id);
                  setDropdownOpen(false);
                }}
              >
                {t(`navigation.${tab.id}`, tab.label)}
              </button>
            ))}
          </div>
        )}
      </div>
    </div>
  );
} 