import { NextRequest, NextResponse } from 'next/server';
import { pool } from '../../../../lib/db';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { user_id, payment_method, items, total_amount } = body;

    // Создаем заказ в статусе "ожидает подтверждения"
    const result = await pool.query(
      'INSERT INTO pending_orders (user_id, payment_method, items, total_amount, status, created_at) VALUES ($1, $2, $3, $4, $5, NOW()) RETURNING id',
      [user_id, payment_method, JSON.stringify(items), total_amount, 'pending']
    );

    const orderId = result.rows[0].id;

    // Отправляем уведомление администратору с ролью 1
    await pool.query(
      'INSERT INTO orders_notifications (user_id, title, message, type, created_at) SELECT u.user_id, $1, $2, $3, NOW() FROM users u WHERE u.role = 1',
      [`Новый заказ`, `Получен новый заказ на сумму ${total_amount} ₪`, 'order']
    );

    return NextResponse.json({ 
      success: true, 
      order_id: orderId,
      message: 'Заказ создан и ожидает подтверждения администратора'
    });

  } catch (error) {
    console.error('Error creating order:', error);
    return NextResponse.json(
      { error: 'Ошибка создания заказа' },
      { status: 500 }
    );
  }
}

export async function PUT(request: NextRequest) {
  try {
    const body = await request.json();
    const { order_id, action, admin_user_id } = body;

    // Получаем информацию о заказе
    const orderResult = await pool.query(
      'SELECT * FROM pending_orders WHERE id = $1',
      [order_id]
    );

    if (orderResult.rows.length === 0) {
      return NextResponse.json(
        { error: 'Заказ не найден' },
        { status: 404 }
      );
    }

    const order = orderResult.rows[0];

    if (action === 'admin_approve') {
      // Администратор подтверждает оплату
      await pool.query(
        'UPDATE pending_orders SET status = $1, admin_approved_at = NOW(), admin_user_id = $2 WHERE id = $3',
        ['admin_approved', admin_user_id, order_id]
      );

      // Отправляем уведомление пользователю с ролью 0
      await pool.query(
        'INSERT INTO orders_notifications (user_id, title, message, type, created_at) SELECT u.user_id, $1, $2, $3, NOW() FROM users u WHERE u.role = 0 AND u.user_id = $4',
        ['Заказ подтвержден', 'Ваш заказ подтвержден администратором', 'order_approved', order.user_id]
      );

      return NextResponse.json({ 
        success: true, 
        message: 'Заказ подтвержден администратором'
      });

    } else if (action === 'user_approve') {
      // Пользователь с ролью 0 подтверждает заказ
      await pool.query(
        'UPDATE pending_orders SET status = $1, user_approved_at = NOW() WHERE id = $2',
        ['user_approved', order_id]
      );

      // Обновляем device_max_value.max_value для устройства пользователя
      const userDevicesResult = await pool.query(
        'SELECT device_id FROM user_devices WHERE user_id = $1',
        [order.user_id]
      );

      if (userDevicesResult.rows.length > 0) {
        const deviceId = userDevicesResult.rows[0].device_id;
        
        // Получаем текущее значение max_value
        const maxValueResult = await pool.query(
          'SELECT max_value FROM device_max_value WHERE device_id = $1',
          [deviceId]
        );

        let newMaxValue = 1000; // Значение по умолчанию
        if (maxValueResult.rows.length > 0) {
          newMaxValue = maxValueResult.rows[0].max_value + 1000; // Увеличиваем на 1000
        }

        // Обновляем или создаем запись
        await pool.query(
          'INSERT INTO device_max_value (device_id, max_value, updated_at) VALUES ($1, $2, NOW()) ON CONFLICT (device_id) DO UPDATE SET max_value = $2, updated_at = NOW()',
          [deviceId, newMaxValue]
        );
      }

      // Создаем документ об оплате через API Sumit
      const userResult = await pool.query(
        'SELECT * FROM users WHERE user_id = $1',
        [order.user_id]
      );

      if (userResult.rows.length > 0) {
        const user = userResult.rows[0];
        
        const payload = {
          Details: {
            Type: 1, // קבלה / חשבונית מס
            Date: new Date().toISOString(),
            Customer: {
              Name: user.f_name || user.email,
              EmailAddress: user.email,
              Phone: user.phone || null,
              City: user.city || null,
              Address: user.address || null,
              ZipCode: user.zipcode || null,
              CompanyNumber: user.p_id || null,
              ID: user.sumit_id || null,
            },
            SendByEmail: {
              EmailAddress: user.email,
              Original: true,
              SendAsPaymentRequest: false,
            },
            DueDate: new Date().toISOString(),
          },
          Items: order.items.map((item: any) => ({
            Quantity: item.quantity,
            UnitPrice: item.unit_price,
            TotalPrice: item.total_price,
            Item: {
              Name: item.name,
              SearchMode: 0,
            }
          })),
          Payments: [
            {
              Amount: order.total_amount,
              Type: order.payment_method === "cash" ? 1 : 4, // 1: Cash, 4: Bank Transfer
              ...(order.payment_method === "cash" ? { Details_Cash: {} } : { Details_BankTransfer: {} })
            }
          ],
          VATIncluded: true,
          Credentials: {
            CompanyID: parseInt(process.env.NEXT_PUBLIC_SUMIT_COMPANY_ID || '0'),
            APIKey: process.env.NEXT_PUBLIC_SUMIT_TOKEN || ''
          }
        };

        try {
          const res = await fetch('https://api.sumit.co.il/accounting/documents/create/', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Accept': 'application/json'
            },
            body: JSON.stringify(payload)
          });

          const data = await res.json();
          
          if (data.Status === 0) {
            // Документ успешно создан
            await pool.query(
              'UPDATE pending_orders SET status = $1, document_created_at = NOW(), sumit_document_id = $2 WHERE id = $3',
              ['completed', data.Data?.DocumentID || null, order_id]
            );
          } else {
            console.error('Error creating Sumit document:', data);
          }
        } catch (error) {
          console.error('Error calling Sumit API:', error);
        }
      }

      return NextResponse.json({ 
        success: true, 
        message: 'Заказ подтвержден и документ создан'
      });
    }

    return NextResponse.json(
      { error: 'Неизвестное действие' },
      { status: 400 }
    );

  } catch (error) {
    console.error('Error updating order:', error);
    return NextResponse.json(
      { error: 'Ошибка обновления заказа' },
      { status: 500 }
    );
  }
}

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const user_id = searchParams.get('user_id');
    const status = searchParams.get('status');

    let whereConditions = [];
    let result;

    if (user_id && status) {
      result = await pool.query(
        'SELECT po.*, u.f_name, u.email FROM pending_orders po JOIN users u ON po.user_id = u.user_id WHERE po.user_id = $1 AND po.status = $2 ORDER BY po.created_at DESC',
        [user_id, status]
      );
    } else if (user_id) {
      result = await pool.query(
        'SELECT po.*, u.f_name, u.email FROM pending_orders po JOIN users u ON po.user_id = u.user_id WHERE po.user_id = $1 ORDER BY po.created_at DESC',
        [user_id]
      );
    } else if (status) {
      result = await pool.query(
        'SELECT po.*, u.f_name, u.email FROM pending_orders po JOIN users u ON po.user_id = u.user_id WHERE po.status = $1 ORDER BY po.created_at DESC',
        [status]
      );
    } else {
      result = await pool.query(
        'SELECT po.*, u.f_name, u.email FROM pending_orders po JOIN users u ON po.user_id = u.user_id ORDER BY po.created_at DESC'
      );
    }

    return NextResponse.json({ 
      orders: result.rows 
    });

  } catch (error) {
    console.error('Error fetching orders:', error);
    return NextResponse.json(
      { error: 'Ошибка получения заказов' },
      { status: 500 }
    );
  }
} 